﻿<# 読取拡張機能スクリプト 2023/11/06
・読取データを様々な方法でチェックするためのスクリプトです。
・RFIDへの読み込み、書き込み処理をします。
・データや排出コマンドを外部プログラムと受け渡しすることができます。
#>
################################################## パラメータ
#重複読取チェック機能
$Duplicate_Flag = $false  #使用する場合は$true、しない場合は$false

#マスタチェック機能
$Master_Path = ""         #マスターデータ ファイルパス。使用しない場合は""
$MasterChk_Pos = 1        #マスターデータ内の読取データ桁位置

#桁数チェック機能
$DataSize = 0             #読取データ桁数。使用しない場合は0

#読取データ編集出力機能（切り取り）
$ResultData_Path = ""     #読取データ編集結果出力ファイルパス。使用しない場合は""
$Cut_Pos = 1              #読取データ切り取り開始位置
$Cut_Len = 1              #読取データ切り取り文字数

#RFID読み取り機能
$CommRfid_R = ""          #RFID読取命令。使用しない場合は"" ※対応していない機器は設定しないでください。
                          #RFID書込時は"F0010`r"、"FU0010`r"などとする。QRとRFIDを同時に読み取る場合などに設定。先の読取データにカンマ区切りで追加される。

#RFID書き込み機能
$CommRfid_W = ""          #RFID書込命令。使用しない場合は""　※対応していない機器は設定しないでください。
                          #RFID書込時は"W00abcd`r"などとする。"W"の場合はダイアログを表示し都度書込データを手入力します。"W00[QrData]`r"でQR読取データを書き込みます。

#読取データ キー入力機能
$ProcessName = ""         #キー入力ターゲットプロセス名。使用しない場合は""。 例）メモ帳:"notepad" 
                          #ターゲットプロセス名を確認するには、プロセスを起動後に Get-Process コマンドレットを実行してProcessNameを確認します。
$TerminateStr = "{ENTER}" #データ入力時終端キー ※特殊キーは、SendKeys文字列を参照してください。  Enter:"{ENTER}"

#オンデマンド機能
$OnData_Path = ""         #直近1件の読取データ出力ファイルパス。使用しない場合は""
$OnComm_Path = ""         #排出コマンド取込ファイルパス。上記データファイルが取込後に削除されたタイミングでこのファイルを監視します。

################################################## 読取拡張機能 有効／無効
#戻り値：OK時は読取データ、NG時は""
Function ReadExteSub($ReadStr) {
    #重複読取チェック
    If ($Duplicate_Flag) {
        If(DuplicateChk $ReadStr) { Return "" }
    }

    #マスタチェック
    If ($Master_Path -ne "") {
        If (MasterChk $ReadStr) { Return "" }
    }

    #桁数チェック
    If ($DataSize -gt 0) {
        If (DataSizeChk $ReadStr) { Return "" }
    }

    #読取データ編集出力
    If ($ResultData_Path -ne "") { EditDataSub $ReadStr }

    #RFID読取処理
    $QrStr = $ReadStr
    If ($CommRfid_R -ne "") {
        $RfidStr = RfidReadSub $ReadStr
        If ($RfidStr -eq "") { Return "" }
        $ReadStr = "$QrStr,$RfidStr"  #QR読取データにRFID読取データを追加
    }

    #RFID書込処理
    If ($CommRfid_W -ne "") {
        If (RfidWriteSub $CommRfid_W $QrStr) { Return "" }
    }

    #キー入力処理
    If ($ProcessName -ne "") {
        If (KeyInputSub $ReadStr) { Return "" }
    }

    #オンデマンド処理
    If ($OnData_Path -ne "") {
        OnDemandSub $ReadStr
        Return ""  #オンデマンド処理時はメインスクリプトの読取OK処理はしない
    }

    Return $ReadStr
}

#---+---+---+---+---+---+---+---+---+---+ エラー時処理
Function DataErrSub($Msg, $ReadStr) {
    $Script:ReadNg_Cnt++  #読取NG枚数カウント
    Write-Host (("{0:D4}" -f $ReadNg_Cnt) + ":$Msg" +":$ReadStr") -ForegroundColor Magenta  #エラーをコンソール表示
    [void]$oWS.Popup($Msg, 2, $WinTitle, 0 + 16)  #メッセージポップアップ

    #エラー排出コマンド
    SendSub (PrintConvSub $CommP1 $PrintNg $ReadNg_Cnt)  #印字データ変換送信
}

################################################## 重複読取チェック処理
#戻り値：重複なし時はfalse、重複あり時はtrue
Function DuplicateChk($ReadStr) {
    foreach ($Str in Get-Content -Path $ReadData_Path) {    #読取データファイル読み込み
        If ($Str -ceq $ReadStr) {
            DataErrSub "重複エラー" $ReadStr
            Return $true
        }
    }
    Return $false
}

################################################## マスタチェック処理
#戻り値：OK時はfalse、NG時はtrue
$MasterChk_Pos = $MasterChk_Pos - 1    #処理上-1しておく

Function MasterChk($ReadStr) {
    foreach ($MasterStr in Get-Content -Path $Master_Path) {    #マスタファイル読み込み
        $Str = CutOutSub $ReadStr $MasterChk_Pos $MasterStr.Length    #読取データをマスタデータにあわせ切り出し
        If ($Str -ceq $MasterStr) {
            Return $false
        }
    }
    DataErrSub "マスタエラー" $ReadStr
    Return $true
}

#---+---+---+---+---+---+---+---+---+---+ データ切り出し
#戻り値：データ切り出し結果
Function CutOutSub($ReadStr, $Pos, $Len) {
    $DataLen = $ReadStr.Length
    If ($DataLen -lt $Pos) {
        $ReadStr = ""
    } Else {
        $OverLen = $Pos + $Len - $DataLen
        If ($OverLen -lt 0) { $OverLen = 0 }
        $ReadStr = $ReadStr.Substring($Pos, $Len - $OverLen)
    }
    Return $ReadStr
}

################################################## 桁数チェック処理
#戻り値：OK時はfalse、NG時はtrue
Function DataSizeChk($ReadStr) {
    If ($ReadStr.Length -ne $DataSize) {
        DataErrSub "桁数エラー" $ReadStr
        Return $true
    }
    Return $false
}

################################################## 読取データ編集出力処理
#戻り値：データ編集結果
$Cut_Pos = $Cut_Pos - 1    #処理上-1しておく

Function EditDataSub($ReadStr) {
    $ResultData = CutOutSub $ReadStr $Cut_Pos $Cut_Len
    Write-Output $ResultData | Out-File -FilePath $ResultData_Path -Encoding Default -Append  #編集データファイル出力
}

################################################## RFID読み取り処理
#戻り値：正常時はRFID読取データを返す、異常時は""
Function RfidReadSub($ReadStr) {
    SendSub $CommRfid_R  #RFID読取コマンド送信
    $StatStr = RecvSub  #ステータス受信
    Switch ($StatStr) {
        "$([char]0x1b)?" {  #読取エラー
            DataErrSub "RFID読取エラー" $ReadStr
            Return ""
        }
        "$([char]0x1b)C" {  #コマンドエラー
            [void]$oWS.Popup("RFID読取コマンドエラー", 2, $WinTitle, 0 + 16)
            ResetSub
            Return ""
        }
    }
    #RFID読取OK
    Return $StatStr  #RFID読取データ
}

################################################## RFID書き込み処理
#戻り値：正常時はfalse、異常時はtrue
Function RfidWriteSub($RF_WriteStr, $ReadStr) {
    If ($RF_WriteStr -eq "W") {
        $RF_WriteStr = "W" + [Microsoft.VisualBasic.Interaction]::InputBox(`
                                "RFID書込データを入力してください。`n開始ブロック 2byte ＋ 書き込み文字列", $WinTitle, "00" + $ReadStr)  #書込データ入力
    } Else {
        $RF_WriteStr = $RF_WriteStr.Replace("[QrData]", $QrStr)  #QR読取データが指定されていれば書き込む
    }

    If ($RF_WriteStr -ne "W") {  #書込データが無い場合はそのまま正常排出
        SendSub $RF_WriteStr  #RFID書込コマンド送信
        $StatStr = RecvSub  #ステータス受信
        Switch ($StatStr) {
            "$([char]0x1b)0" {  #正常応答
                Return $false
            }
            "$([char]0x1b)W" {  #書込エラー
                DataErrSub "RFID書込エラー" $ReadStr
                Return $true
            }
            "$([char]0x1b)C" {  #コマンドエラー
                [void]$oWS.Popup("RFID書込コマンドエラー", 2, $WinTitle, 0 + 16)
                ResetSub
                Return $true
            }
        }
        #MainSub $StatStr
        Return $true
    }
    Return $false
}

################################################## 読取データ キー入力処理
#ターゲットプロセスをアクティブにして読取データを自動入力します。
#戻り値：正常時はfalse、異常時はtrue
$ps = Get-Process | Where-Object {$_.Name -match $ProcessName}  #ターゲットプロセス取得

Function KeyInputSub($ReadStr) {
    $Flag = 0
    Do {
        $Flag = 0
        Try {
            [Microsoft.VisualBasic.Interaction]::AppActivate($ps.ID)  #ターゲットプロセスをアクティブにする
        } Catch {
            $Flag = $oWS.Popup("キー入力するターゲットプロセスが見つかりません。`n"`
                             + "プロセスが起動しているかを確認してください。`n"`
                             + "読取を終了する場合は キャンセル を押してください。", 0, $WinTitle, 5 + 16)
            $ps = Get-Process | Where-Object {$_.Name -match $ProcessName}  #ターゲットプロセス取得
        }
    } While($Flag -eq 4)  #再試行

    If ($Flag -eq 0) {
        [System.Windows.Forms.SendKeys]::SendWait($ReadStr + $TerminateStr)  #キー出力
        Return $false
    } Else {
        DataErrSub "キー入力エラー" $ReadStr
        Remove-Item -Path $Active_FileName  #実行中ファイル削除 スクリプト終了
        Return $true
    }
}

################################################## オンデマンド処理
#メディアを1枚読み取りする毎にファイル出力し、外部プログラムとファイルによる読み取りデータおよびコマンドの受け渡しを行います。
#ファイルの作成、削除は必ず下記の手順にて行ってください。手順が異なるとデータまたはコマンドが正しく受け取れない場合があります。[]内は外部プログラムでの処理です。
#  機器読取OK ⇒ データファイル作成 ⇒ コマンドファイル削除 ⇒ [データファイル取込] ⇒ [コマンドファイル作成] ⇒ [データファイル削除] ⇒ コマンドファイル取込  
$FileChkWait = 10  #ファイル監視待ち時間 ms

If ($OnData_Path -ne "") {
    If ( -not (Test-Path $OnComm_Path)) { New-Item $OnComm_Path }  #スタート時コマンドファイルが無ければ作成する
}

Function OnDemandSub($ReadStr) {
    Write-Output $ReadStr | Out-File -FilePath $OnData_Path -Encoding Default  #データファイル作成（上書き）
    Remove-Item $OnComm_Path  #コマンドファイル削除
    While (Test-Path $OnData_Path) {  #データファイルが取込まれるまで待機
        Start-Sleep -m $FileChkWait
        If (-not (Test-Path $Active_FileName)) { Return }   #実行中ファイルが存在しなければ処理終了
    }
    $Comm = Get-Content $OnComm_Path  #コマンドファイル取込
    $Comm = $Comm.Replace('`r', "`r")  #`r文字をCRコードに変換

    #Pコマンド確認
    $P0 = $CommP0
    If ($P0 -match "`r$") { $P0 = $P0.Substring(0, $P0.Length - 1) }  #末尾CRがあれば削除
    $P1 = $CommP1
    If ($P1 -match "`r$") { $P1 = $P1.Substring(0, $P1.Length - 1) }  #末尾CRがあれば削除
    
    If ($Comm -like "$P0*") {  #OK排出確認
        $Script:ReadOk_Cnt++  #読取OK枚数カウント
        Write-Host (("{0:D4}" -f $ReadOk_Cnt) + ":" + $ReadStr)  #読取データをコンソール表示
        $Cnt = $ReadOk_Cnt
        Write-Output $ReadStr | Out-File -FilePath $ReadData_Path -Encoding Default -Append  #読取データをファイルへ追記
    }
    ElseIf ($Comm -like "$P1*") {  #NG排出確認
        $Script:ReadNg_Cnt++  #読取NG枚数カウント
        Write-Host (("{0:D4}" -f $ReadNg_Cnt) + ":DataError:" + $ReadStr) -ForegroundColor Magenta  #読取データをコンソール表示
        $Cnt = $ReadNg_Cnt
    }
    #RFID書込コマンド確認
    ElseIf ($Comm -like "W*") {
        If (RfidWriteSub $Comm $ReadStr) { Return }  #RFID書込
        $Script:ReadOk_Cnt++  #読取OK枚数カウント
        Write-Host (("{0:D4}" -f $ReadOk_Cnt) + ":" + $ReadStr)  #読取データをコンソール表示
        $Cnt = $ReadOk_Cnt
        Write-Output $ReadStr | Out-File -FilePath $ReadData_Path -Encoding Default -Append  #読取データをファイルへ追記
        $Comm = $CommP0  #正常排出コマンド
    }

    #排出コマンド送信
    If ($Comm -match "`r$") { $Comm = $Comm.Substring(0, $Comm.Length - 1) }  #末尾CRがあれば削除
    SendSub (PrintConvSub "" $Comm $Cnt)  #印字データ変換送信
}
